<?php

namespace App\Filament\Pages\Settings;

use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class EditProfile extends Page implements HasForms
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-user';
    protected static string $view = 'filament.pages.settings.edit-profile';
    protected static ?string $navigationGroup = 'Settings'; // Groups it in the sidebar
    protected static ?string $title = 'Edit Profile';

    // This holds the form data
    public ?array $data = [];

    // This method runs when the page is loaded
    public function mount(): void
    {
        $this->form->fill(auth()->user()->attributesToArray());
    }

    // This defines the form's schema
    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Profile Information')
                    ->description('Update your name and profile picture.')
                    ->schema([
                        FileUpload::make('avatar')
                            ->image()
                            ->imageEditor()
                            ->avatar() // Makes the preview circular
                            ->disk('public') // Uses the public disk
                            ->directory('avatars'), // Stores files in 'storage/app/public/avatars'
                        TextInput::make('name')
                            ->required(),
                    ]),

                Section::make('Update Password')
                    ->description('Ensure your account is using a long, random password to stay secure.')
                    ->schema([
                        TextInput::make('current_password')
                            ->password()
                            ->required()
                            ->currentPassword(), // Special validation rule
                        TextInput::make('new_password')
                            ->password()
                            ->required()
                            ->confirmed() // Requires 'new_password_confirmation' to match
                            ->different('current_password'),
                        TextInput::make('new_password_confirmation')
                            ->password()
                            ->required(),
                    ]),
            ])
            ->statePath('data'); // Binds the form to our $data property
    }

    // This method is called when the form is submitted
    public function updateProfile()
    {
        $validatedData = $this->form->getState();
        $user = auth()->user();

        // Update name and avatar
        $user->update([
            'name' => $validatedData['name'],
            'avatar' => $validatedData['avatar'],
        ]);

        // Update password if a new one is provided
        if (!empty($validatedData['new_password'])) {
            $user->update([
                'password' => Hash::make($validatedData['new_password']),
            ]);
             session()->put('password_hash_' . Auth::getDefaultDriver(), $user->password);
        }
        Auth::login($user);
        // Send a success notification
        Notification::make()
            ->title('Profile updated successfully!')
            ->success()
            ->send();
            
        // Refresh the page to see changes
        // The new, reliable line
return redirect(static::getUrl());
    }
}
